﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using GE.World;
using GE.TimeClock;
using GE.Manager;

using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

using System;

namespace GE.World.Entities
{
    /// <summary>
    /// A bom thrown by Bombman
    /// </summary>
    class BombmanBombEntity : WorldEntity
    {
        enum eState
        {
            eStateThrown,
            eStateFall,
            eStateCount
        }

        /// <summary>
        /// Bomb's inner state
        /// </summary>
        eState _innerState;

        /// <summary>
        /// Texture id
        /// </summary>
        int _iIdTexture;

        /// <summary>
        /// Sprite id
        /// </summary>
        int _iIdSprite;

        /// <summary>
        /// Collision box
        /// </summary>
        DynamicShapeRectangle _shape;

        /// <summary>
        /// Bomb target
        /// </summary>
        private Vector2 _v2Target;

        /// <summary>
        /// Max height the bomb can reach.
        /// </summary>
        const float MAX_HEIGHT = 100;

        /// <summary>
        /// Time the bomb takes to do its displacement on the x axis.
        /// </summary>
        const int LIFETIME_X_DISPLACEMENT = 500;

        /// <summary>
        /// bomb's birth time
        /// </summary>
        int _iBirthTime;

        /// <summary>
        /// Bomb's initial position
        /// </summary>
        Vector2 _v2StartPosition;

        /// <summary>
        /// Distance between the initial position and the target
        /// </summary>
        float _fDistanceOfTarget;

        /// <summary>
        /// Explosion animation
        /// </summary>
        int _iIdAnimationExplosion;

        #region Properties

        /// <summary>
        /// Set the bomb's target
        /// </summary>
        public Vector2 Target { set { _v2Target = value; } }

        #endregion

        /// <summary>
        /// Constructor
        /// </summary>
        public BombmanBombEntity()
            : base()
        {
            _iIdTexture = -1;
            _iIdSprite = -1;
        }

        /// <summary>
        /// Initialise the entity
        /// </summary>
        public override void init()
        {
            _iIdTexture = Visu.Instance.loadTilset("bombmansheet.xml");
            _iIdSprite = Visu.Instance.getSpriteId(_iIdTexture, "bomb");
            _iIdAnimationExplosion = Visu.Instance.getAnimationID("Big_Explosion");

            int iWidth = Visu.Instance.getSpriteWidth(_iIdTexture, _iIdSprite);
            int iHeight = Visu.Instance.getSpriteHeight(_iIdTexture, _iIdSprite);
            _shape = Physics.Physics.Instance.createDynamicRectangle(iWidth, iHeight, Vector2.Zero, this);
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;
            _shape._bCollisionEnable = false;
        }

        /// <summary>
        /// Activator
        /// </summary>
        public override void activate()
        {
            _shape._bCollisionEnable = true;
            _shape._v2position = Position;
            _innerState = eState.eStateThrown;
            _iBirthTime = Clock.instance.millisecs;
            _v2StartPosition = _v2Position;
            _fDistanceOfTarget = _v2Target.X - _v2Position.X;
            base.activate();
        }

        /// <summary>
        /// Update
        /// </summary>
        public override void update()
        {
            switch (_innerState)
            {
                case eState.eStateThrown:
                    updateStateThrown();
                    break;

                case eState.eStateFall:
                    updateStateFall();
                    break;
            }

            updateCollision();
        }

        /// <summary>
        /// Update the state going up
        /// </summary>
        private void updateStateThrown()
        {
            float timeElapsed = Clock.instance.millisecs - _iBirthTime;
            if (timeElapsed > LIFETIME_X_DISPLACEMENT)
            {
                _innerState = eState.eStateFall;
            }

            //y interpolation
            float ratio = timeElapsed / LIFETIME_X_DISPLACEMENT;
            float delta = (float)MathHelper.Pi * ratio;
            _v2Position.Y = _v2StartPosition.Y - (float)Math.Sin(delta) * MAX_HEIGHT;

            //x interpolation
            _v2Position.X = _v2StartPosition.X + _fDistanceOfTarget * ratio;

            _shape._v2position = _v2Position;
        }

        /// <summary>
        /// Update the state going down
        /// </summary>
        private void updateStateFall()
        {
            _v2Position.Y += 10;

            _shape._v2position = _v2Position;
        }

        /// <summary>
        /// Update the collision
        /// </summary>
        private void updateCollision()
        {
            CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlayer | (int)ePhysicGroup.ePhysicPlatform);
            if (res == null) return;

            die();
            if(res.Entity.Side == eSide.eSidePlayer)
                World.Instance.PlayerEntity.hurt(Damages);
        }

        /// <summary>
        /// Render
        /// </summary>
        public override void render()
        {
            Visu.Instance.displaySprite(_iIdTexture, _iIdSprite, ScreenPosition);
        }

        public override void die()
        {
            _shape._bCollisionEnable = false;
            _bActive = false;
            Manager.ExplosionManager.Instance.activate(_iIdAnimationExplosion, Position);
        }
    }
}
